/******************************************************************************
HPCalcSerial.c
Copyright (C)2007-2008 Allen Wan
License: GPLv3 or any later version

ver. 20080217
	polarity of invalidSleepEnable switch was changed to default to 0

******************************************************************************/

#define F_CPU 1000000L

#define _FORCEOFF_PIN	2
#define _ENABLE_PIN	0
#define FORCEON_PIN	2

#define INVALID_SLEEP	1
#define CALC_OFF_SLEEP	4
#define SERIAL_OFF_SLEEP	5

#include <stdio.h>
#include <util/delay.h>
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/sleep.h>

int turnON(int type);
int turnOFF(int type);
int invalid(void);
int doInvalidSleep (void);
int doCalcSerialSleep (void);

int mirrorBit(void);
void watch(void);

unsigned int i;
int sleepType = 0;
int invalidSleepEnable;
int deGlitchWait;

ISR(PCINT0_vect)
{	
}

int turnON(int type)
{
	if(type == 1)
	{
		PORTB = (1<<_FORCEOFF_PIN) | (1<<_ENABLE_PIN);
	}
	else
	{
		PORTB = (1<<_FORCEOFF_PIN) | (0<<_ENABLE_PIN);
	}

	if(invalidSleepEnable)
	{
		PORTA = 0;
	}
	else
	{
		PORTA = _BV(FORCEON_PIN);
	}
	return 0;
}

int turnOFF(int type)
{
	if(type == INVALID_SLEEP)
	{
		PORTB = (0<<_FORCEOFF_PIN) | (1<<_ENABLE_PIN);
		PORTA = (0<<FORCEON_PIN);
		return INVALID_SLEEP;
	}
	else if(type == CALC_OFF_SLEEP)
	{
		PORTB = (0<<_FORCEOFF_PIN) | (1<<_ENABLE_PIN);
		PORTA = (0<<FORCEON_PIN);
		return CALC_OFF_SLEEP;
	}
	else if(type == SERIAL_OFF_SLEEP)
	{
		PORTB = (0<<_FORCEOFF_PIN) | (1<<_ENABLE_PIN);
		PORTA &= ~_BV(FORCEON_PIN);
		return SERIAL_OFF_SLEEP;
	}
	else
	{
		PORTB &= ~_BV(_FORCEOFF_PIN);
		PORTB |= _BV(_ENABLE_PIN);
		PORTA &= ~_BV(FORCEON_PIN);
		return 0;
	}
}

int invalid(void)
{
//the actual signal is invalid bar
//so response is inverted
	if(PINA & (1<<PA5))
	{
		return 0;
	}
	//if invalidSleepEnable is set to zero,
	//function should always return zero.
	return invalidSleepEnable;	
}

int doInvalidSleep(void)
{
	cli();
	turnOFF(INVALID_SLEEP);
	PCMSK0 = (0<<PCINT7)|(0<<PCINT6)|(1<<PCINT5)|(0<<PCINT4)|(0<<PCINT3)|(0<<PCINT2)|(0<<PCINT1)|(0<<PCINT0);
	GIMSK = (0<<INT0)|(0<<PCIE1)|(1<<PCIE0);
	sleepType = INVALID_SLEEP;
	sleep_enable();
	sei();
	sleep_cpu();
	return 0;
}

int doCalcSerialSleep(void)
{
	cli();
	if(PINA&(1<<PA4))
	{
		PORTA = (1<<PA1);
		PCMSK0 = (0<<PCINT7)|(0<<PCINT6)|(1<<PCINT5)|(1<<PCINT4)|(0<<PCINT3)|(0<<PCINT2)|(1<<PCINT1)|(0<<PCINT0);
		GIMSK = (0<<INT0)|(0<<PCIE1)|(1<<PCIE0);
		turnOFF(SERIAL_OFF_SLEEP);
		sleepType = SERIAL_OFF_SLEEP;
	}
	else
	{
		PORTA = 0;
		PCMSK0 = (0<<PCINT7)|(0<<PCINT6)|(0<<PCINT5)|(1<<PCINT4)|(0<<PCINT3)|(0<<PCINT2)|(0<<PCINT1)|(0<<PCINT0);
		GIMSK = (0<<INT0)|(0<<PCIE1)|(1<<PCIE0);
		turnOFF(CALC_OFF_SLEEP);
		sleepType = CALC_OFF_SLEEP;
	}
	sleep_enable();
	sei();
	sleep_cpu();
	return 0;
}

int mirrorBit(void)
{	
	if(((PINA&0b00000010)<<2) != (PINA&0b00001000))
	{
		if(((PINA&0b00000010)<<2) != (PINA&0b00001000))
		{
			if(((PINA&0b00000010)<<2) != (PINA&0b00001000))
			{
				return 1;
			}
		}
	}
	return 0;
}

void watch(void)
{
	while(1)
	{
		if(sleepType == CALC_OFF_SLEEP)
		{

			if(deGlitchWait)
			{
				turnON(1);
				_delay_ms(8);
			}
			else
			{
				turnON(0);
			}
			sleepType = 0;
		}
		else if(sleepType == SERIAL_OFF_SLEEP)
		{
			if(PINA&(1<<PA4))
			{
				if(deGlitchWait)
				{
				DDRA = _BV(PA3)|_BV(PA2);//bit should already be set to 0
				//PA2 should already be an output. The decision not to do a "bit OR"
				//type assign is speed driven. This section of code is very speed
				//critical
				turnON(1);
				_delay_ms(8);
				DDRA = _BV(PA2);
					//this assumes that turnON clears the PA1 pullup
				}
				else
				{
					turnON(0);
				}
				sleepType = 0;
			}
			else
			{
				if(PINA&1<<PA1)
				{
						//this case should NOT occur
					sleepType = 0;
						//assumes that turnON in main loop
						//clears the PA1 pullup
				}
				else
				{
					doCalcSerialSleep();
						//assumes that doCalcSerialSleep
						//clears the PA1 pullup
				}
			}
		}
		else if(sleepType == INVALID_SLEEP)
		{
			sleepType = 0;
		}
		else
		{
			if(invalid())
			{
				doInvalidSleep();
			}
			else
			{
				turnON(0);
				if(mirrorBit())
				{
					turnOFF(0);
					PORTA |= _BV(PA1);
					_delay_us(10);
					if(PINA&(1<<PA1))
					{
						turnON(0);
					}
					else
					{
						doCalcSerialSleep();
					}
				}
			}
		}
	}
}

int main(void)
{
	PORTA = 0b00000000;
	PORTB = (0<<_FORCEOFF_PIN) | (1<<_ENABLE_PIN);
	DDRA = 0b00000000;
	DDRB = 0b00000101;
	MCUCR = (0<<PUD)|(1<<SE)|(1<<SM1)|(0<<SM0)|(0<<ISC01)|(0<<ISC00);
	PCMSK0 = (0<<PCINT7)|(0<<PCINT6)|(0<<PCINT5)|(1<<PCINT4)|(0<<PCINT3)|(0<<PCINT2)|(1<<PCINT1)|(0<<PCINT0);

	PORTA = (1<<PA0);//Enables pullup resistor on PA0
	_delay_ms(8);
	if(PINA & (1<<PA0))
	{
		//If PA0 is high (and thus NOT shorted to GND; default)
		//invalidSleepEnable is set to 0
		invalidSleepEnable = 0;
	}
	else
	{
		//If PA0 is low (due to pullup resistor being pulled down to GND)
		//invalidSleepEnable is set to 1
		invalidSleepEnable = 1;
	}
	
	PORTA = (1<<PA7);//Enables pullup resistor on PA7
	DDRA = (1<<PA6);//Sets PA6 to output (which means it's driving LOW)
	_delay_ms(1);//Checking immediately without delay will miss the short due to settling
	if(PINA & (1<<PA7))
	{
		//If PA7 is high (and thus NOT shorted to PA6 which is driving LOW)
		//deGlitchWait is set to 1 (default)
		deGlitchWait = 1;
	}
	else
	{
		//If PA7 is low (due to pullup resistor being pulled down)
		//deGlitchWait is set to 0
		deGlitchWait = 0;
	}
	
	PORTA = 0b00000000;
	DDRA = (1<<PA2);//used to control FORCEON

	PORTA = _BV(PA1);
	_delay_us(10);
	if(PINA&(1<<PA1))
	{
		//turnON(0);
	}
	else
	{
		doCalcSerialSleep();
	}
	
	watch();
	return 0;
}
